﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using Advent.aiDevices;

namespace Advent.Controls
{
    public partial class AI7280TelInt : UserControl
    {
        private AI7280_CO_Simulator.TelephoneInterface TelInt;

        public AI7280TelInt()
        {
            InitializeComponent();

            cbo_SourceMode.Items.Clear();
            cbo_SourceMode.Items.Add(AI7280_CO_Simulator.TelIntSourceMode.Constant_Current);
            cbo_SourceMode.Items.Add(AI7280_CO_Simulator.TelIntSourceMode.Constant_Voltage);

            cbo_MeasurementPoint.Items.Clear();
            cbo_MeasurementPoint.Items.Add(AI7280_CO_Simulator.TelIntMeasurementPoint.Inside_Pair);
            cbo_MeasurementPoint.Items.Add(AI7280_CO_Simulator.TelIntMeasurementPoint.Outside_Pair);

            cbo_MeasurementRange.Items.Clear();
            cbo_MeasurementRange.Items.Add(AI7280_CO_Simulator.TelIntMeasurementRange.Normal_Range);
            cbo_MeasurementRange.Items.Add(AI7280_CO_Simulator.TelIntMeasurementRange.High_Range);

            this.EnabledChanged += new EventHandler(AI7280TelInt_EnabledChanged);

            txt_OffHookThresh.Validating += new CancelEventHandler(TextBoxValidation);
            txt_SourceCurrent.Validating += new CancelEventHandler(TextBoxValidation);
            txt_SourceVoltage.Validating += new CancelEventHandler(TextBoxValidation);

            txt_OffHookThresh.KeyUp += new KeyEventHandler(txt_OffHookThresh_KeyUp);
            txt_SourceCurrent.KeyUp += new KeyEventHandler(txt_OffHookThresh_KeyUp);
            txt_SourceVoltage.KeyUp += new KeyEventHandler(txt_OffHookThresh_KeyUp);
        }

        #region Text Box Event Handling


        void txt_OffHookThresh_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode != Keys.Return) return;
            try
            {
                ParseTextBoxes();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }

        void TextBoxValidation(object sender, CancelEventArgs e)
        {
            try
            {
                ParseTextBoxes();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                e.Cancel = true;
                RefreshControls();
            }
        }
        #endregion

        void ParseTextBoxes()
        {

            double V;
            double I;
            double Thresh;

            if (!double.TryParse(txt_SourceVoltage.Text, out V)) throw new Exception("Invalid source voltage");
            if (!double.TryParse(txt_SourceCurrent.Text, out I)) throw new Exception("Invalid source voltage");
            if (!double.TryParse(txt_OffHookThresh.Text, out Thresh)) throw new Exception("Invalid threshold current");

            TelInt.SourceVoltage = DCVoltage.InVolts(V);
            TelInt.SourceCurrent = DCCurrent.InMilliAmps(I);
            TelInt.OffHookCurrentThreshold = DCCurrent.InMilliAmps(Thresh);
        }


        void AI7280TelInt_EnabledChanged(object sender, EventArgs e)
        {
            timer_Update.Enabled = Enabled;
        }

        public void SetTelInt(AI7280_CO_Simulator.TelephoneInterface TelInt)
        {
            this.TelInt = TelInt;

            // Load all the available fixed impedances into the combo boxes
            cbo_ACImpedance.Items.Clear();
            cbo_Balance.Items.Clear();
            foreach (Impedance Z in TelInt.FixedImpedancesAvailable)
            {
                cbo_ACImpedance.Items.Add(Z);
                cbo_Balance.Items.Add(Z);
            }
            RefreshControls();
            UpdateStatus();
        }

        /// <summary>
        /// This loads each of the controls with the contents of the telephone interface settings
        /// </summary>
        private void RefreshControls()
        {
            try {
                if (TelInt == null) return;
                txt_SourceVoltage.Text = TelInt.SourceVoltage.Value.ToString();
                txt_SourceCurrent.Text = TelInt.SourceCurrent.ValueInMilliAmps.ToString();

                cbo_SourceMode.SelectedItem = TelInt.SourceMode;
                cbo_ACImpedance.SelectedItem = TelInt.ACImpedance;
                cbo_Balance.SelectedItem = TelInt.Balance;
                
                txt_OffHookThresh.Text = TelInt.OffHookCurrentThreshold.ValueInMilliAmps.ToString();

                cbo_MeasurementPoint.SelectedItem = TelInt.MeasurementPoint;
                cbo_MeasurementRange.SelectedItem = TelInt.MeasurementRange;
            }catch {}

        }

        /// <summary>
        /// This updates the status display controls
        /// </summary>
        private void UpdateStatus()
        {
            try
            {
                if (TelInt == null) return;

                string Status = "Telephone Interface";

                if (TelInt.IsOffHook) Status += " (Off hook)";
                if (TelInt.IsDisconnected) Status += " (Disconnected)";
                gb_TelInt.Text = Status;
            }
            catch
            {
            }
        }



        private void bnt_Connect_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.Connect();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }

        private void btn_Disconnect_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.Disconnect();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }

        private void btn_Reverse_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.Reverse();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }

        private void timer_Update_Tick(object sender, EventArgs e)
        {
            try
            {
                UpdateStatus();
            }
            catch { }
        }

        private void lbl_ResetToDefaults_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.ResetToDefaults();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
            finally
            {
                RefreshControls();
                UpdateStatus();

            }



        }

        private void cbo_SourceMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                TelInt.SourceMode = (AI7280_CO_Simulator.TelIntSourceMode) cbo_SourceMode.SelectedItem;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }

        private void cbo_ACImpedance_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                TelInt.ACImpedance = (Impedance)cbo_ACImpedance.SelectedItem;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }

        private void cbo_Balance_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                TelInt.Balance = (Impedance) cbo_Balance.SelectedItem;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }

        private void cbo_MeasurementPoint_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                TelInt.MeasurementPoint = (AI7280_CO_Simulator.TelIntMeasurementPoint) cbo_MeasurementPoint.SelectedItem;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }

        private void cbo_MeasurementRange_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                TelInt.MeasurementRange = (AI7280_CO_Simulator.TelIntMeasurementRange) cbo_MeasurementRange.SelectedItem;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }
    }
}
